from PyQt6.QtWidgets import (
    QDialog, QVBoxLayout, QLabel, QComboBox,
    QPushButton, QFileDialog, QHBoxLayout,
    QSpinBox, QLineEdit
)
from PyQt6.QtCore import Qt, QThread, pyqtSignal
import os
import time
import subprocess
from TTS_Progress import TTSProgressDialog
import pygame
import textwrap

class TTSWorker(QThread):
    progress_updated = pyqtSignal(int, int, str)  # current, total, status
    finished = pyqtSignal()
    error_occurred = pyqtSignal(str)

    def __init__(self, text, model_path, output_dir, filename, format, split_chars):
        super().__init__()
        self.text = text
        self.model_path = model_path
        self.output_dir = output_dir
        self.filename = filename
        self.format = format
        self.split_chars = split_chars
        self.is_cancelled = False
        self.piper_path = r"D:\TTS\piper\piper.exe"  # Укажите ваш путь к piper.exe

    def split_text_by_paragraphs(self, text, max_chars):
        """Разделяет текст на части по абзацам, не превышающие max_chars символов"""
        paragraphs = text.split('\n')
        chunks = []
        current_chunk = ""
        
        for para in paragraphs:
            # Если текущий абзац + новый превышает лимит
            if len(current_chunk) + len(para) + 1 > max_chars:
                if current_chunk:  # Если есть что добавить
                    chunks.append(current_chunk)
                    current_chunk = ""
                
                # Если отдельный абзац слишком длинный, разбиваем его принудительно
                if len(para) > max_chars:
                    forced_chunks = textwrap.wrap(para, width=max_chars)
                    chunks.extend(forced_chunks[:-1])
                    current_chunk = forced_chunks[-1]
                else:
                    current_chunk = para
            else:
                if current_chunk:
                    current_chunk += "\n" + para
                else:
                    current_chunk = para
        
        if current_chunk:
            chunks.append(current_chunk)
            
        return chunks

    def run(self):
        try:
            # Разбиваем текст на части по абзацам
            text_chunks = self.split_text_by_paragraphs(self.text, self.split_chars)
            total_chunks = len(text_chunks)
            
            for i, chunk in enumerate(text_chunks, 1):
                if self.is_cancelled:
                    break
                    
                # Обновляем прогресс
                self.progress_updated.emit(i, total_chunks, f"Обработка части {i}/{total_chunks}")
                
                # Генерируем имя файла с номером части
                chunk_filename = f"{self.filename}_{i:03d}"
                temp_file = os.path.join(self.output_dir, f"temp_{chunk_filename}.wav")
                
                # Конвертация части текста
                cmd = [self.piper_path, "--model", self.model_path, "--output_file", temp_file]
                
                process = subprocess.Popen(
                    cmd,
                    stdin=subprocess.PIPE,
                    stdout=subprocess.PIPE,
                    stderr=subprocess.PIPE,
                    text=True,
                    encoding='utf-8'
                )
                
                process.communicate(input=chunk)
                
                if not os.path.exists(temp_file):
                    raise Exception(f"Piper не создал выходной файл для части {i}")
                
                # Конвертация формата
                final_file = os.path.join(self.output_dir, f"{chunk_filename}.{self.format}")
                if self.format != "wav":
                    self.convert_audio(temp_file, final_file, self.format)
                    os.remove(temp_file)
                else:
                    os.rename(temp_file, final_file)
                
            if not self.is_cancelled:
                self.finished.emit()

        except Exception as e:
            self.error_occurred.emit(str(e))

    def convert_audio(self, input_file, output_file, format):
        """Конвертация аудио в другой формат с помощью ffmpeg"""
        try:
            cmd = [
                'ffmpeg',
                '-i', input_file,
                '-b:a', '128k',  # Битрейт по умолчанию
                output_file
            ]
            result = subprocess.run(cmd, capture_output=True, text=True, timeout=30)
            if result.returncode != 0:
                raise Exception(f"FFmpeg ошибка: {result.stderr}")
        except subprocess.TimeoutExpired:
            raise Exception("FFmpeg: Таймаут конвертации")
        except FileNotFoundError:
            raise Exception("FFmpeg не найден. Установите ffmpeg в систему.")
        except Exception as e:
            raise Exception(f"Ошибка конвертации аудио: {str(e)}")

class TTSDialog(QDialog):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("Piper TTS Settings")
        self.setFixedSize(450, 350)
        self.main_window = parent
        self.output_dir = os.path.expanduser(r"C:\Users\1\Desktop")
        self.model_path = r"D:\TTS\Ruslan\ru_RU-ruslan-medium.onnx"  # Путь к модели
        self.init_ui()
        pygame.mixer.init()

    def init_ui(self):
        layout = QVBoxLayout()

        # Формат аудио
        layout.addWidget(QLabel("Audio Format:"))
        self.format_combo = QComboBox()
        self.format_combo.addItems(["wav", "mp3", "aac"])
        layout.addWidget(self.format_combo)

        # Настройки битрейта
        bitrate_layout = QHBoxLayout()
        bitrate_layout.addWidget(QLabel("Bitrate:"))
        
        self.bitrate_type = QComboBox()
        self.bitrate_type.addItems(["Constant", "Variable"])
        bitrate_layout.addWidget(self.bitrate_type)
        
        self.bitrate_spin = QSpinBox()
        self.bitrate_spin.setRange(32, 320)
        self.bitrate_spin.setValue(128)
        bitrate_layout.addWidget(self.bitrate_spin)
        layout.addLayout(bitrate_layout)

        # Разделение по символам
        layout.addWidget(QLabel("Split by characters (0 for no split):"))
        self.split_spin = QSpinBox()
        self.split_spin.setRange(0, 100000)
        self.split_spin.setValue(3000)
        self.split_spin.setSingleStep(500)
        layout.addWidget(self.split_spin)

        # Имя файла
        layout.addWidget(QLabel("Output Filename:"))
        self.filename_edit = QLineEdit("output")
        layout.addWidget(self.filename_edit)

        # Выбор папки
        self.folder_btn = QPushButton("Select Output Folder")
        self.folder_btn.clicked.connect(self.select_output_folder)
        layout.addWidget(self.folder_btn)

        # Кнопка конвертации
        self.convert_btn = QPushButton("Convert to Speech")
        self.convert_btn.clicked.connect(self.start_conversion)
        layout.addWidget(self.convert_btn)

        self.setLayout(layout)
        self.progress_dialog = None
        self.worker = None

    def select_output_folder(self):
        folder = QFileDialog.getExistingDirectory(
            self,
            "Select Output Folder",
            self.output_dir
        )
        if folder:
            self.output_dir = folder

    def start_conversion(self):
        if self.main_window and hasattr(self.main_window, 'text_browser'):  # ✅ Исправлено
            text = self.main_window.text_browser.toPlainText()           
            if text.strip():
                split_chars = self.split_spin.value()               
                self.progress_dialog = TTSProgressDialog(self)
                self.progress_dialog.start()
                
                self.worker = TTSWorker(
                    text=text,
                    model_path=self.model_path,
                    output_dir=self.output_dir,
                    filename=self.filename_edit.text(),
                    format=self.format_combo.currentText(),
                    split_chars=split_chars if split_chars > 0 else float('inf')  # Если 0 - не разбиваем
                )
                
                self.worker.progress_updated.connect(self.update_progress)
                self.worker.finished.connect(self.on_conversion_finished)
                self.worker.error_occurred.connect(self.on_conversion_error)
                
                self.worker.start()

    def update_progress(self, current, total, status):
        if self.progress_dialog:
            progress = int((current / total) * 100)
            self.progress_dialog.progress.setValue(progress)
            self.progress_dialog.status_label.setText(status)

    def on_conversion_finished(self):
        if self.progress_dialog:
            self.progress_dialog.progress.setValue(100)
            self.progress_dialog.status_label.setText("Готово!")
            self.progress_dialog.close()

    def on_conversion_error(self, error_msg):
        if self.progress_dialog:
            self.progress_dialog.status_label.setText(f"Ошибка: {error_msg}")
            self.progress_dialog.close()
        self.worker = None

    def closeEvent(self, a0) -> None:
        if self.worker and self.worker.isRunning():
            self.worker.is_cancelled = True
            self.worker.wait(1000)
        if self.progress_dialog:
            self.progress_dialog.close()
        
        # Проверяем, что a0 не None
        if a0 is not None:
            a0.accept()