from pydub import AudioSegment
import os
import shutil
import time
import wave
import struct

class AudioSplitter:
    def __init__(self, output_dir, filename):
        self.output_dir = output_dir
        self.filename = filename

    def process_audio(self, input_path, format, split_minutes, progress_callback=None):
        try:
            # Конвертируем RAW WAV в стандартный WAV сначала
            normalized_wav = os.path.join(self.output_dir, f"normalized_{os.path.basename(input_path)}")
            self._convert_raw_to_standard_wav(input_path, normalized_wav)
            
            # Теперь работаем с нормализованным файлом
            if split_minutes > 0:
                return self._split_audio(normalized_wav, split_minutes, format, progress_callback)
            else:
                return self._save_single_file(normalized_wav, format)
                
        except Exception as e:
            print(f"Ошибка обработки: {str(e)}")
            return None
        finally:
            # Удаляем временный файл
            if os.path.exists(normalized_wav):
                os.remove(normalized_wav)

    def _convert_raw_to_standard_wav(self, input_path, output_path):
        """Конвертирует RAW PCM WAV в стандартный WAV"""
        try:
            # Читаем RAW WAV (предполагаем параметры Piper: 16bit, 22.05kHz, mono)
            with open(input_path, 'rb') as f:
                raw_data = f.read()
            
            # Создаем стандартный WAV
            with wave.open(output_path, 'wb') as wav_file:
                wav_file.setnchannels(1)  # моно
                wav_file.setsampwidth(2)  # 16bit = 2 байта
                wav_file.setframerate(22050)  # частота дискретизации
                wav_file.writeframes(raw_data)
                
        except Exception as e:
            raise Exception(f"Ошибка конвертации RAW WAV: {str(e)}")

    def _split_audio(self, input_path, split_minutes, format, progress_callback):
        """Нарезка аудио с проверкой формата"""
        try:
            ext = os.path.splitext(input_path)[1][1:].lower()
            audio = AudioSegment.from_file(input_path, format=ext)
            chunk_ms = split_minutes * 60 * 1000
            total_ms = len(audio)
            
            if total_ms <= chunk_ms:
                return self._save_single_file(input_path, format)

            chunks = []
            for start in range(0, total_ms, chunk_ms):
                end = min(start + chunk_ms, total_ms)
                chunks.append(audio[start:end])

            results = []
            for i, chunk in enumerate(chunks, 1):
                part_name = f"{self.filename}_part{i}.{format}"
                output_path = os.path.join(self.output_dir, part_name)
                chunk.export(output_path, format=format)
                results.append(output_path)
                
                if progress_callback:
                    progress_callback(i, len(chunks))

            return results
            
        except Exception as e:
            raise Exception(f"Ошибка нарезки: {str(e)}")

    def _save_single_file(self, input_path, format):
        """Простое сохранение с конвертацией формата"""
        output_path = os.path.join(self.output_dir, f"{self.filename}.{format}")
        audio = AudioSegment.from_file(input_path)
        audio.export(output_path, format=format)
        return [output_path]