import sys
import os
from PyQt6.QtWidgets import (
    QApplication, QMainWindow, QTextBrowser,
    QPushButton, QFileDialog, QVBoxLayout, QWidget
)
from PyQt6.QtCore import QSettings
import chardet
from bs4 import BeautifulSoup
from ebooklib import epub
from striprtf.striprtf import rtf_to_text
import docx
from TTS_Engine import TTSDialog

class DocumentReader(QMainWindow):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Document Reader")
        self.setGeometry(100, 100, 800, 600)
        self.file_path = None
        self.settings = QSettings("MyApp", "DocumentReader")
        self.last_dir = self.settings.value("last_dir", "", str)
        self.init_ui()
    
    def init_ui(self):
        self.text_browser = QTextBrowser()
        self.btn_open = QPushButton("Open Document")
        self.btn_open.clicked.connect(self.open)
        
        self.tts_btn = QPushButton("Convert to Speech")
        self.tts_btn.clicked.connect(self.open_tts_dialog)
        
        layout = QVBoxLayout()
        layout.addWidget(self.btn_open)
        layout.addWidget(self.text_browser)
        layout.addWidget(self.tts_btn)
        
        container = QWidget()
        container.setLayout(layout)
        self.setCentralWidget(container)
    
    def open(self):
        
        file_path, _ = QFileDialog.getOpenFileName(
            self, "Select Document", self.last_dir,
            "Documents (*.txt *.fb2 *.epub *.html *.htm *.rtf *.docx)"
        )
        if file_path:
            self.file_path = file_path
            self.last_dir = os.path.dirname(file_path)
            self.settings.setValue("last_dir", self.last_dir)
            self.display()
    
    def display(self):
        if not self.file_path:
            return
            
        try:
            if self.file_path.lower().endswith('.txt'):
                with open(self.file_path, "rb") as f:
                    raw = f.read(10000)
                    encoding = chardet.detect(raw)["encoding"]
                with open(self.file_path, "r", encoding=encoding) as f:
                    self.text_browser.setPlainText(f.read())
            
            elif self.file_path.lower().endswith('.fb2'):
                with open(self.file_path, "rb") as f:
                    soup = BeautifulSoup(f.read(), 'xml')
                    for binary in soup.find_all('binary'):
                        binary.decompose()
                    self.text_browser.setPlainText(soup.get_text())
            
            elif self.file_path.lower().endswith('.epub'):
                book = epub.read_epub(self.file_path)
                full_text = []
                for item in book.get_items():
                    if isinstance(item, epub.EpubHtml):
                        content = item.get_content()
                        # Проверяем тип и декодируем только если это байты
                        if isinstance(content, bytes):
                            content = content.decode('utf-8')
                        full_text.append(content)
                self.text_browser.setHtml("<br>".join(full_text))
            
            elif self.file_path.lower().endswith(('.html', '.htm')):
                with open(self.file_path, "r", encoding='utf-8') as f:
                    self.text_browser.setHtml(f.read())
            
            elif self.file_path.lower().endswith('.rtf'):
                with open(self.file_path, "rb") as f:
                    try:
                        text = rtf_to_text(f.read().decode('latin-1'))
                        self.text_browser.setPlainText(text)
                    except Exception as e:
                        self.text_browser.setPlainText(f"RTF Error: {str(e)}")
            
            elif self.file_path.lower().endswith('.docx'):
                doc = docx.Document(self.file_path)
                full_text = [para.text for para in doc.paragraphs]
                self.text_browser.setPlainText('\n'.join(full_text))
            
            else:
                self.text_browser.setPlainText("Unsupported file format")
                
        except Exception as e:
            self.text_browser.setPlainText(f"Error: {str(e)}")
    
    def open_tts_dialog(self):
        if hasattr(self, 'text_browser'):
            dialog = TTSDialog(self)
            dialog.exec()

if __name__ == "__main__":
    app = QApplication(sys.argv)
    window = DocumentReader()
    window.show()
    sys.exit(app.exec())