import sys
import os

# 1. Сначала ВСЕ системные библиотеки
import chardet
from bs4 import BeautifulSoup
from ebooklib import epub
from striprtf.striprtf import rtf_to_text
import docx

# 2. Потом PyTorch (ДО PyQt6!)
try:
    import torch
    print(f"PyTorch загружен: {torch.__version__}")
    TORCH_AVAILABLE = True
except Exception as e:
    print(f"PyTorch не доступен: {e}")
    TORCH_AVAILABLE = False

# 3. Только потом PyQt6
from PyQt6.QtWidgets import (
    QApplication, QMainWindow, QTextBrowser,
    QPushButton, QFileDialog, QVBoxLayout, QWidget
)
from PyQt6.QtCore import QSettings

# 4. В конце ваш модуль
from TTS_Engine import TTSDialog

class DocumentReader(QMainWindow):
    def __init__(self):
        super().__init__()
        self.setWindowTitle("Document Reader")
        self.setGeometry(250, 50, 800, 600)
        self.file_path = None
        self.settings = QSettings("MyApp", "DocumentReader")
        # Безопасное чтение: если значения нет, будет пустая строка
        self.last_dir = self.settings.value("last_dir", "", type=str) # Исправлено: добавлен type=str
        self.init_ui()
    
    def init_ui(self):
        self.text_browser = QTextBrowser()
        self.btn_open = QPushButton("Open Document")
        self.btn_open.clicked.connect(self.open)
        
        self.tts_btn = QPushButton("Convert to Speech")
        self.tts_btn.clicked.connect(self.open_tts_dialog)
        
        layout = QVBoxLayout()
        layout.addWidget(self.btn_open)
        layout.addWidget(self.text_browser)
        layout.addWidget(self.tts_btn)
        
        container = QWidget()
        container.setLayout(layout)
        self.setCentralWidget(container)
    
    def open(self):
        file_path, _ = QFileDialog.getOpenFileName(
            self, "Select Document", self.last_dir,
            "Documents (*.txt *.fb2 *.epub *.html *.htm *.rtf *.docx)"
        )
        if file_path:
            self.file_path = file_path
            self.last_dir = os.path.dirname(file_path)
            self.settings.setValue("last_dir", self.last_dir)
            self.display()
    
    def display(self):
        if not self.file_path:
            return
            
        try:
            if self.file_path.lower().endswith('.txt'):
                # Для текстовых файлов более надежное определение кодировки
                with open(self.file_path, "rb") as f:
                    raw_data = f.read()
                    # Проверяем BOM для UTF-8
                    if raw_data.startswith(b'\xef\xbb\xbf'):
                        encoding = 'utf-8-sig'
                        raw_data = raw_data[3:]
                    else:
                        result = chardet.detect(raw_data[:100000])  # Проверяем первые 100КБ
                        encoding = result['encoding'] if result['encoding'] else 'utf-8'
                
                with open(self.file_path, "r", encoding=encoding, errors='replace') as f:
                    self.text_browser.setPlainText(f.read())
            
            elif self.file_path.lower().endswith('.fb2'):
                # FB2: используем более аккуратное извлечение текста
                with open(self.file_path, "r", encoding="utf-8", errors='ignore') as f:
                    soup = BeautifulSoup(f.read(), 'lxml-xml')  # Явно указываем парсер
                    
                    # Удаляем все бинарные данные и скрипты
                    for tag in soup.find_all(['binary', 'script', 'style']):
                        tag.decompose()
                    
                    # Извлекаем текст только из body
                    body = soup.find('body')
                    if body:
                        text = body.get_text(separator='\n', strip=True)
                    else:
                        text = soup.get_text(separator='\n', strip=True)
                    
                    # Очищаем множественные переносы строк
                    lines = [line.strip() for line in text.split('\n') if line.strip()]
                    self.text_browser.setPlainText('\n'.join(lines))
            
            elif self.file_path.lower().endswith('.epub'):
                # EPUB: правильный способ чтения
                book = epub.read_epub(self.file_path)
                full_text = []
                
                # Получаем все элементы в правильном порядке
                items = list(book.get_items())
                
                for item in items:
                    # Берем только текстовые/HTML элементы
                    if item.get_type() == epub.ITEM_DOCUMENT:
                        content = item.get_content()
                        
                        if content:
                            # Если контент в байтах, декодируем
                            if isinstance(content, bytes):
                                try:
                                    content = content.decode('utf-8')
                                except UnicodeDecodeError:
                                    # Пробуем другие кодировки
                                    try:
                                        content = content.decode('cp1251')
                                    except:
                                        content = "[Binary content, cannot decode]"
                            
                            # Парсим HTML, извлекаем текст
                            soup = BeautifulSoup(content, 'html.parser')
                            
                            # Удаляем скрипты и стили
                            for tag in soup.find_all(['script', 'style', 'nav']):
                                tag.decompose()
                            
                            text = soup.get_text(separator='\n', strip=True)
                            if text:
                                full_text.append(text)
                
                self.text_browser.setPlainText('\n\n'.join(full_text))
            
            elif self.file_path.lower().endswith(('.html', '.htm')):
                # HTML: используем BeautifulSoup для очистки
                with open(self.file_path, "r", encoding='utf-8', errors='replace') as f:
                    soup = BeautifulSoup(f.read(), 'html.parser')
                    
                    # Удаляем скрипты и стили
                    for tag in soup.find_all(['script', 'style']):
                        tag.decompose()
                    
                    text = soup.get_text(separator='\n', strip=True)
                    self.text_browser.setPlainText(text)
            
            elif self.file_path.lower().endswith('.rtf'):
                # RTF: более надежная обработка
                with open(self.file_path, "rb") as f:
                    raw_data = f.read()
                    
                    # Пробуем разные кодировки
                    encodings_to_try = ['utf-8', 'cp1251', 'cp1252', 'latin-1']
                    
                    for encoding in encodings_to_try:
                        try:
                            decoded = raw_data.decode(encoding)
                            text = rtf_to_text(decoded)
                            self.text_browser.setPlainText(text)
                            break
                        except:
                            continue
                    else:
                        # Если ни одна кодировка не подошла
                        self.text_browser.setPlainText(
                            "RTF Error: Cannot decode file. Try converting to another format."
                        )
            
            elif self.file_path.lower().endswith('.docx'):
                doc = docx.Document(self.file_path)
                full_text = []
                
                # Извлекаем текст из параграфов
                for para in doc.paragraphs:
                    if para.text.strip():
                        full_text.append(para.text)
                
                # Также пытаемся извлечь текст из таблиц
                for table in doc.tables:
                    for row in table.rows:
                        row_text = []
                        for cell in row.cells:
                            if cell.text.strip():
                                row_text.append(cell.text)
                        if row_text:
                            full_text.append(' | '.join(row_text))
                
                self.text_browser.setPlainText('\n'.join(full_text))
            
            else:
                self.text_browser.setPlainText("Unsupported file format")
                
        except Exception as e:
            import traceback
            error_details = traceback.format_exc()
            self.text_browser.setPlainText(f"Critical Error: {str(e)}\n\nDetails:\n{error_details}")
    
    def open_tts_dialog(self):
        # Проверяем, есть ли текст для обработки
        if hasattr(self, 'text_browser') and self.text_browser.toPlainText().strip():
            dialog = TTSDialog(self)
            dialog.exec()
        else:
            # Можно добавить всплывающее предупреждение
            self.text_browser.setPlainText("No text to convert. Please open a document first.")

if __name__ == "__main__":
    app = QApplication(sys.argv)
    window = DocumentReader()
    window.show()
    sys.exit(app.exec())